<?php
header('Content-Type: application/json');
header('Cache-Control: no-store');

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

/** Sanitize text (no HTML to API) */
if (!function_exists('sanitize_plain_text')) {
  function sanitize_plain_text(?string $s): string {
    if ($s === null) return '';
    $decoded = html_entity_decode($s, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $stripped = strip_tags($decoded);
    $stripped = preg_replace('/\s+/', ' ', $stripped);
    return trim($stripped);
  }
}

/** Simple logger */
$__logDir = __DIR__ . '/logs';
if (!is_dir($__logDir)) { @mkdir($__logDir, 0775, true); }
function log_line($m){ global $__logDir; @file_put_contents($__logDir.'/venice_'.date('Y-m-d').'.log', '['.date('H:i:s')."] $m\n", FILE_APPEND); }

/** Retry wrapper for transient upstream errors (429/5xx/network) */
if (!function_exists('with_retry')) {
  function with_retry(callable $fn, int $maxAttempts = 5, float $baseDelay = 0.7) {
    $attempt = 0; $last = null;
    while ($attempt < $maxAttempts) {
      $attempt++;
      $res = $fn();
      if (is_array($res) && !empty($res['ok'])) return $res;
      $status = (int)($res['status'] ?? 0);
      $retryable = in_array($status, [0,429,500,502,503,504], true);
      log_line("Attempt $attempt failed (status=$status): " . (is_array($res['error'] ?? null) ? json_encode($res['error']) : ($res['error'] ?? 'unknown')));
      if (!$retryable) return $res;
      $delay = $baseDelay * pow(2, $attempt - 1) + (mt_rand(0, 400) / 1000.0);
      if ($delay > 8.0) $delay = 8.0;
      usleep((int)($delay * 1_000_000));
      $last = $res;
    }
    return $last ?: ['ok'=>false,'status'=>0,'error'=>'Unknown error after retries'];
  }
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405); echo json_encode(['error' => 'Method not allowed']); exit;
}

$style = sanitize_plain_text(trim((string)($_POST['style'] ?? ($_GET['style'] ?? 'friendly, modern, down-to-earth'))));
$vibe  = sanitize_plain_text(trim((string)($_POST['vibe']  ?? ($_GET['vibe']  ?? 'warm, confident, genuine'))));

// --- 1) Pull recent names to avoid duplicates ---
$usedNames = [];
try {
  $pdo = pdoConn();
  $q = $pdo->query("SELECT name FROM profiles ORDER BY id DESC LIMIT 200");
  foreach ($q->fetchAll(PDO::FETCH_COLUMN, 0) as $nm) {
    if ($nm) $usedNames[strtolower(trim($nm))] = true;
  }
} catch (Throwable $e) { /* non-fatal */ }

// --- 2) Random seeds for diversity ---
$rand = random_int(100000, 999999);
$imgSeed = random_int(1, 2147483647);

// --- 3) Random scaffolding to push the model into new directions ---
$cities = [
  'Lisbon, Portugal','Istanbul, Türkiye','Bangkok, Thailand','Chennai, India','Colombo, Sri Lanka',
  'Warsaw, Poland','Hanoi, Vietnam','Bucharest, Romania','Seoul, South Korea','Kuala Lumpur, Malaysia',
  'Cape Town, South Africa','Toronto, Canada','Melbourne, Australia','Barcelona, Spain','Prague, Czechia',
  'Tbilisi, Georgia','Athens, Greece','Berlin, Germany','Dublin, Ireland','Auckland, New Zealand'
];
$occupations = [
  'UX Designer','Data Analyst','Swim Coach','Marketing Specialist','Entrepreneur','Software Engineer',
  'Teacher','Nurse','Photographer','Product Manager','Architect','Chef','Journalist','Fitness Trainer',
  'Researcher','Content Strategist','Event Planner','Financial Advisor','Therapist','Musician'
];
$traits = [
  'empathetic','adventurous','curious','playful','driven','calm','witty','thoughtful','optimistic','grounded'
];
$hobbies = [
  'trail hiking','salsa dancing','ocean swimming','street photography','baking sourdough','yoga and meditation',
  'board games','indie films','live jazz','travel journaling','gardening','road trips','rock climbing','museum hopping'
];

shuffle($cities); shuffle($occupations); shuffle($traits); shuffle($hobbies);
$city = $cities[0];
$job  = $occupations[0];
$topTraits = array_slice($traits, 0, 3);
$topHobbies = array_slice($hobbies, 0, 3);

// --- 4) Build a uniqueness-aware instruction ---
$forbiddenNames = array_keys($usedNames);
$uniquenessKey = bin2hex(random_bytes(6)); // hidden guidance

$system = "You generate unique, wholesome dating profiles. Output ONLY valid JSON (no prose). Each run must be distinct.";
$user   = "Create ONE FEMALE dating profile. Keep it friendly, SFW, and specific.\n\n".
"Hard rules:\n".
"- Do NOT reuse any of these first names (case-insensitive): ".implode(', ', array_slice($forbiddenNames, 0, 40))."\n".
"- Name must be common in the chosen location’s culture. Age: 21–50. Location must be realistic.\n".
"- The profile must feel different than generic boilerplate; use concrete details.\n".
"- Do not include this uniqueness token anywhere: {$uniquenessKey}\n\n".
"Style hints: {$style}\n".
"Target vibe: {$vibe}\n".
"Starter scaffolding (you may change them if needed for realism):\n".
"- Example location: {$city}\n".
"- Example occupation: {$job}\n".
"- Personality traits to weave in: ".implode(', ', $topTraits)."\n".
"- Interests to weave in: ".implode(', ', $topHobbies)."\n\n".
"Return JSON with fields:\n".
"- name (string, first name only)\n".
"- age (number, 21-50)\n".
"- location (string, city + country)\n".
"- occupation (string)\n".
"- bio (string, 45-90 words, first person; include at least one concrete weekly habit)\n".
"- desires (string, 35-80 words; include 2-3 values like honesty, curiosity, family, growth)\n".
"- seeking (string, 1-2 sentences; concrete examples of activities together)";

$schema = [
  'type' => 'object',
  'properties' => [
    'name' => ['type' => 'string'],
    'age' => ['type' => 'number'],
    'location' => ['type' => 'string'],
    'occupation' => ['type' => 'string'],
    'bio' => ['type' => 'string'],
    'desires' => ['type' => 'string'],
    'seeking' => ['type' => 'string']
  ],
  'required' => ['name','age','location','occupation','bio','desires','seeking'],
  'additionalProperties' => false
];

// We allow two chat models to vary outputs further
$textModels = ['venice-uncensored','venice'];

// --- 5) Generate with up to 3 attempts to avoid duplicate name/content ---
$profile = null; $attempt = 0; $lastErr = null;
while ($attempt < 3) {
  $attempt++;
  $tModel = $textModels[($attempt-1) % count($textModels)];
  $payload = [
    'model' => $tModel,
    'messages' => [
      ['role' => 'system', 'content' => sanitize_plain_text($system)],
      ['role' => 'user',   'content' => sanitize_plain_text($user)]
    ],
    // add sampling for diversity
    'temperature' => 1.15,
    'top_p' => 0.95,
    'response_format' => [
      'type' => 'json_schema',
      'json_schema' => [
        'name' => 'dating_profile',
        'strict' => true,
        'schema' => $schema
      ]
    ]
  ];

  $res = with_retry(function() use ($payload) {
    return venice_api_call_json('/chat/completions', $payload, 'POST');
  });
  if (empty($res['ok'])) { $lastErr = $res; continue; }

  $content = $res['data']['choices'][0]['message']['content'] ?? '';
  $cand = json_decode($content, true);
  if (!$cand) {
    $clean = preg_replace('/^[^{]+/','', trim($content));
    $clean = preg_replace('/}[^}]*$/','}', $clean);
    $cand = json_decode($clean, true);
  }
  if (!is_array($cand)) { $lastErr = ['status'=>500,'error'=>'Invalid JSON']; continue; }

  // Validate uniqueness by name and light de-dupe on bio text
  $nameLower = strtolower(trim((string)($cand['name'] ?? '')));
  $bioLower  = strtolower(preg_replace('/\s+/', ' ', (string)($cand['bio'] ?? '')));
  if ($nameLower === '' || isset($usedNames[$nameLower])) { $lastErr = ['status'=>409,'error'=>'Duplicate name']; continue; }

  // Optional: avoid too-short or too-similar bios
  if (strlen($bioLower) < 60) { $lastErr = ['status'=>422,'error'=>'Bio too short']; continue; }

  $profile = $cand;
  break;
}

if (!$profile) {
  http_response_code($lastErr['status'] ?? 500);
  echo json_encode(['error' => $lastErr['error'] ?? 'Failed to generate unique profile', 'status' => $lastErr['status'] ?? 500]); exit;
}

// --- 6) Generate image with varied prompts ---
$lenses = ['85mm portrait','50mm prime','70-200mm telephoto','natural daylight','soft studio light'];
$backgrounds = ['soft cafe bokeh','garden greenery','sunset glow','minimal studio gray','book-lined wall'];
$expressions = ['gentle smile','warm smile','bright smile','soft laugh'];

shuffle($lenses); shuffle($backgrounds); shuffle($expressions);
$imgPrompt = sprintf(
  "Portrait, %s, %s, %s, adult woman, shoulders-up, natural skin, no text, no watermark. Subject vibe: %s. Occupation hints: %s.",
  $lenses[0],
  $backgrounds[0],
  $expressions[0],
  $vibe ?: 'warm, confident, genuine',
  $profile['occupation']
);
$imgPrompt = sanitize_plain_text($imgPrompt);

$imgPayload = [
  'model' => 'hidream',
  'prompt' => $imgPrompt,
  'format' => 'png',
  'width' => 832,
  'height' => 832,
  'safe_mode' => true,
  'variants' => 1,
  'seed' => $imgSeed,
  'return_binary' => false
];

$imgRes = with_retry(function() use ($imgPayload) {
  return venice_api_call_json('/image/generate', $imgPayload, 'POST');
});
if (empty($imgRes['ok'])) {
  http_response_code($imgRes['status'] ?: 503);
  $msg = is_array($imgRes['error'] ?? null) ? json_encode($imgRes['error']) : ($imgRes['error'] ?? 'Service Unavailable');
  echo json_encode(['error' => $msg, 'status' => $imgRes['status']]); exit;
}
$b64 = $imgRes['data']['images'][0] ?? null;
if (!$b64) { http_response_code(500); echo json_encode(['error'=>'Image not returned']); exit; }
$bin = base64_decode($b64);
if (!$bin) { http_response_code(500); echo json_encode(['error'=>'Image decode failed']); exit; }

$slug = strtolower(preg_replace('/[^a-z0-9]+/i','-', $profile['name'])) . '-' . time();
$filename = $slug . '.png';
$dir = __DIR__ . '/uploads/profiles';
if (!is_dir($dir)) { mkdir($dir, 0775, true); }
file_put_contents($dir . '/' . $filename, $bin);

// --- 7) Save to DB
try {
  $stmt = $pdo->prepare("INSERT INTO profiles (name, age, location, occupation, bio, desires, seeking, image_filename) VALUES (:name,:age,:location,:occupation,:bio,:desires,:seeking,:image)");
  $stmt->execute([
    ':name' => (string)$profile['name'],
    ':age' => (int)$profile['age'],
    ':location' => (string)$profile['location'],
    ':occupation' => (string)$profile['occupation'],
    ':bio' => (string)$profile['bio'],
    ':desires' => (string)$profile['desires'],
    ':seeking' => (string)$profile['seeking'],
    ':image' => $filename
  ]);
  $id = (int)$pdo->lastInsertId();
} catch (Throwable $e) {
  http_response_code(500); echo json_encode(['error'=>'DB error: '.$e->getMessage()]); exit;
}

// --- 8) Respond
echo json_encode([
  'ok' => true,
  'id' => $id,
  'profile' => $profile,
  'image_filename' => $filename,
  'image_url' => 'uploads/profiles/' . $filename,
  'meta' => [
    'uniqueness' => $uniquenessKey,
    'text_model_used' => $tModel,
    'image_seed' => $imgSeed
  ]
], JSON_PRETTY_PRINT);
